class NewsRenderer {
  constructor() {
    this.format = null;
    this.dimensions = null;
    this.ratio = null;
    this.fontSize = 55;
    this._imageStyles = {};
    this.debounceLog = this.debounce(this.logImageStyles, 300);

  }

  /* FUNÇÕES DE AJUSTE DE IMAGEM */
  imageDragging() {
    const mediaContainer = document.getElementById('template-container');
    const backgroundMedia = document.getElementById('background_media');
    const mediaElement = backgroundMedia.querySelector('img, video');
    const containerRatio = this.ratio;
    const imageRatio = this.backgroundAspectRatio();

    mediaElement.style.position = 'absolute';  // Certifique-se de que a imagem está posicionada de forma absoluta

    if (containerRatio <= imageRatio) {
      mediaElement.style.height = '100%';
    } else {
      mediaElement.style.width = '100%';
    }

    let isDragging = false;
    let startX, startY, initialLeft, initialTop;

    mediaContainer.addEventListener('mousedown', (e) => {
      isDragging = true;
      startX = e.clientX;
      startY = e.clientY;
      initialLeft = parseInt(window.getComputedStyle(mediaElement).marginLeft);
      initialTop = parseInt(window.getComputedStyle(mediaElement).marginTop);
      mediaContainer.style.cursor = 'grabbing';
    });

    mediaContainer.addEventListener('mousemove', (e) => {
      if (isDragging) {
        const dx = e.clientX - startX;
        const dy = e.clientY - startY;
        mediaElement.style.marginLeft = `${initialLeft + dx}px`;
        mediaElement.style.marginTop = `${initialTop + dy}px`;
        this.imageStyleData();
      }
    });

    mediaContainer.addEventListener('mouseup', () => {
      isDragging = false;
      mediaContainer.style.cursor = 'grab';
    });

    mediaContainer.addEventListener('mouseleave', () => {
      isDragging = false;
      mediaContainer.style.cursor = 'grab';
    });
  }

  imageZooming() {
    const mediaContainer = document.getElementById('template-container');
    const backgroundMedia = document.getElementById('background_media');
    const mediaElement = backgroundMedia.querySelector('img, video');
    mediaElement.style.position = 'absolute';  // Certifique-se de que a imagem está posicionada de forma absoluta

    let scale = 1;  // Escala inicial da imagem
    const zoomFactor = 0.1;  // Fator de zoom

    mediaContainer.addEventListener('wheel', (e) => {
      e.preventDefault(); // Impede o comportamento padrão de rolagem

      if (e.deltaY < 0) {
        // Rolagem para cima, aumenta o zoom
        scale += zoomFactor;
      } else {
        // Rolagem para baixo, diminui o zoom
        scale = Math.max(0.1, scale - zoomFactor);  // Evita que a escala fique negativa ou muito pequena
      }

      mediaElement.style.transform = `scale(${scale})`;
      mediaElement.style.transformOrigin = 'center center';  // Define o ponto de origem do zoom no centro da imagem
      this.imageStyleData();
    });
  }

  resetImageStylesOnDoubleClick() {
    const mediaContainer = document.getElementById('template-container');
    const backgroundMedia = document.getElementById('background_media');
    const mediaElement = backgroundMedia.querySelector('img, video');

    mediaContainer.addEventListener('dblclick', () => {
      // Remove os estilos aplicados
      mediaElement.style.marginLeft = '';  // Reseta a posição horizontal
      mediaElement.style.marginTop = '';   // Reseta a posição vertical
      mediaElement.style.transform = '';  // Reseta o zoom
      mediaElement.style.transformOrigin = '';  // Reseta o ponto de origem do zoom
      this.imageStyleData();
    });
  }

  alignImageWithArrowKeys() {
    const mediaContainer = document.getElementById('template-container');
    const backgroundMedia = document.getElementById('background_media');
    const mediaElement = backgroundMedia.querySelector('img, video');

    document.addEventListener('keydown', (e) => {
      const containerHeight = mediaContainer.clientHeight;
      const imageHeight = mediaElement.getBoundingClientRect().height;

      const containerWidth = mediaContainer.clientWidth;
      const imageWidth = mediaElement.getBoundingClientRect().width;

      if (e.key === 'ArrowUp' && e.ctrlKey) {
        //obter margem top atual
        const currentTop = parseInt(mediaElement.style.marginTop) || 0;
        mediaElement.style.marginTop = `${currentTop + 1}px`;
      } else if (e.key === 'ArrowUp') {
        const marginTop = (imageHeight - containerHeight);
        mediaElement.style.marginTop = `${marginTop}px`;
      }

      if (e.key === 'ArrowDown' && e.ctrlKey) {
        //obter margem top atual
        const currentTop = parseInt(mediaElement.style.marginTop) || 0;
        mediaElement.style.marginTop = `${currentTop - 1}px`;
      } else if (e.key === 'ArrowDown') {
        const marginTop = -(imageHeight - containerHeight);
        mediaElement.style.marginTop = `${marginTop}px`;
      }

      if (e.key === 'ArrowLeft' && e.ctrlKey) {
        //obter margem left atual
        const currentLeft = parseFloat(mediaElement.style.marginLeft) || 0;
        mediaElement.style.marginLeft = `${currentLeft + 1}px`;
      } else if (e.key === 'ArrowLeft') {
        const marginLeft = (imageWidth - containerWidth);
        mediaElement.style.marginLeft = `${marginLeft}px`;
      }

      if (e.key === 'ArrowRight' && e.ctrlKey) {
        //obter margem left atual
        const currentLeft = parseFloat(mediaElement.style.marginLeft) || 0;
        mediaElement.style.marginLeft = `${currentLeft - 1}px`;
      } else if (e.key === 'ArrowRight') {
        const marginLeft = -(imageWidth - containerWidth);
        mediaElement.style.marginLeft = `${marginLeft}px`;
      }

      if (e.key === 'ArrowRight' || e.key === 'ArrowLeft' || e.key === 'ArrowUp' || e.key === 'ArrowDown') {
        e.preventDefault();
        this.imageStyleData();
      }
    });
  }

  // Getter para acessar imageStyles
  get imageStyles() {
    return this._imageStyles;
  }

  // Setter para alterar imageStyles e emitir o log com debounce
  set imageStyles(newStyles) {
    this._imageStyles = newStyles;
    this.debounceLog(); // Chama a função log com debounce
  }

  // Função para logar o valor de imageStyles
  logImageStyles() {
    console.log('imageStyles changed:', this._imageStyles);
    window.parent.postMessage('image-styles', '*');
  }

  // Função debounce para evitar múltiplos logs consecutivos
  debounce(func, wait) {
    let timeout;
    return () => {
      clearTimeout(timeout);
      timeout = setTimeout(() => func.call(this), wait);
    };
  }

  // Método para atualizar imageStyles
  imageStyleData() {
    const mediaContainer = document.getElementById('template-container');
    const backgroundMedia = document.getElementById('background_media');
    const mediaElement = backgroundMedia.querySelector('img, video');

    const imgScale = mediaElement.style.transform.match(/scale\(([^)]+)\)/) ?? null;

    const imageTop = parseFloat(mediaElement.style.marginTop) || 0;
    const imageLeft = parseFloat(mediaElement.style.marginLeft) || 0;

    const containerHeight = mediaContainer.clientHeight;
    const imageHeight = mediaElement.getBoundingClientRect().height.toFixed(1);

    const containerWidth = mediaContainer.clientWidth;
    const imageWidth = mediaElement.getBoundingClientRect().width.toFixed(1);

    const maxTop = parseFloat((imageHeight - containerHeight).toFixed(1));
    const maxLeft = parseFloat((imageWidth - containerWidth).toFixed(1));

    const offsetBottom = imageTop < -maxTop ? -(imageTop + maxTop) : 0;
    const offsetTop = imageTop > maxTop ? imageTop - maxTop : 0;
    const offsetRight = imageLeft < -maxLeft ? -(imageLeft + maxLeft) : 0;
    const offsetLeft = imageLeft > maxLeft ? imageLeft - maxLeft : 0;

    // Cria um novo objeto com os estilos calculados
    const imageStyles = {
      marginTop: imageTop,
      marginLeft: imageLeft,
      offsetBottom,
      offsetTop,
      offsetRight,
      offsetLeft,
      imgScale: imgScale ? imgScale[1] : 1,
      container: {
        height: containerHeight,
        width: containerWidth,
        ratio: containerWidth / containerHeight
      }
    };

    // Atualiza a propriedade this.imageStyles, acionando o setter
    this.imageStyles = imageStyles;
  }

  /* FINAL DAS FUNÇÕES DE AJUSTE DE IMAGEM */

  backgroundAspectRatio() {
    const backgroundMedia = document.getElementById('background_media');
    const mediaElement = backgroundMedia.querySelector('img, video'); // Seleciona img ou video

    if (mediaElement.tagName.toLowerCase() === 'img') {
      // Para imagens
      const imgWidth = mediaElement.naturalWidth;
      const imgHeight = mediaElement.naturalHeight;
      return imgWidth / imgHeight;
    } else if (mediaElement.tagName.toLowerCase() === 'video') {
      // Para vídeos
      const videoWidth = mediaElement.videoWidth;
      const videoHeight = mediaElement.videoHeight;
      return videoWidth / videoHeight;
    } else {
      // Caso não seja nem imagem nem vídeo
      console.error('Elemento não é uma imagem nem um vídeo.');
      return null;
    }
  }

  screenFormat(dimensions) {

    const windowWidth = dimensions?.width || window.innerWidth;
    const windowHeight = dimensions?.height || window.innerHeight;
    const ratio = windowWidth / windowHeight;

    this.ratio = ratio

    if (ratio < 1) {
      this.format = 'vertical';
    } else if (ratio >= 1 && ratio <= 4) {
      this.format = 'horizontal';
    } else {
      this.format = 'wide';
    }
  }

  countLines(element) {
    const lineHeight = parseFloat(getComputedStyle(element).lineHeight);
    const totalHeight = element.scrollHeight;
    return Math.round(totalHeight / lineHeight);
  }

  adjustContentInContainer() {
    const descriptionContainer = document.getElementById('description-container');
    const descriptionText = document.getElementById('description-text');

    const containerHeight = Math.ceil(descriptionContainer.offsetHeight);
    const textHeight = Math.ceil(descriptionText.offsetHeight);

    const fontSize = Number(window.getComputedStyle(descriptionText).fontSize.replace('px', ''));
    console.log(fontSize)
    console.log(this.fontSize)
    if(containerHeight < textHeight) {
      const newSize = fontSize - 1;
      //console.log(newSize);
      descriptionText.style.fontSize = newSize + 'px';
      setTimeout(() => {
        this.adjustContentInContainer();
      }, 10);
    }
  
  }

  adjustContentFontSize() {
    const element = document.getElementById('description-text');
    const numberOfLines = this.countLines(element);
    const lines = this.format === 'vertical' ? 5 : this.format === 'wide' ? 8 : 2;
    if (numberOfLines > lines) {
      this.fontSize--;
      element.style.fontSize = this.fontSize + 'px';
      this.adjustContentFontSize();
    }
  }

  adjustTitleFontSize() {
    const container = document.getElementById('description-container');
    const element = document.getElementById('title');
    const containerWidth = container.offsetWidth;
    const elementWidth = element.offsetWidth;
    if (elementWidth > containerWidth) {
      this.fontSize--;
      element.style.fontSize = this.fontSize + 'px';
      this.adjustTitleFontSize();
    }
  }

  verticalQrCodePosition(marginBottom) {
    const container = document.getElementById('information-bar');
    const title = document.getElementById('title');
    const titleHeight = title.offsetHeight;
    const containerHeight = container.offsetHeight + marginBottom + (marginBottom / 5) - titleHeight;

    const qrCodeElement = document.getElementById('qr-code');
    if (this.format === 'vertical') {
      qrCodeElement.style.setProperty('bottom', `${containerHeight}px`);
      qrCodeElement.style.setProperty('top', `unset`);
      qrCodeElement.style.setProperty('z-index', 99);
    }
  }

  readyToStart(data) {
    const duration = data?.item?.duration ?? 10;
    const progressBar = document.getElementById('progress-bar');
    progressBar.classList.add('start-animation');
    progressBar.style.setProperty('animation-duration', parseInt(duration) + 's');

    const backgroundMedia = document.getElementById('background_media');
    const video = backgroundMedia.querySelector('video');

    if (video) {
      video.controls = false;
      video.currentTime = 0;
      video.play();
    }
  }

  receivedTemplateData(data) {
    if (!data.item) return; // Caso a mensagem venha sem o objeto
    const item = data.item;
    //const imagesPath = 'http://localhost/elemidia_v4/cache';
    const object = {
      ...item,
      qr_code: item.qr_code_absolute,
      file: item.imagem,
      logo_rounded: item.logo_absolute,
      logo_square: item.logo_absolute
    };

    const mapElementId = {
      'file': { 'id': 'background_media' },
      'title': { 'id': 'title' },
      'text': { 'id': 'description-text' },
      'credits': { 'id': 'credits' },
      'image_credits': { 'id': 'image_credits' },
      'qr_code': { 'id': 'qr-code' },
      'logo_rounded': { 'id': 'brand-rounded' },
      'logo_square': { 'id': 'brand-square' }
    };

    const marginBottom = data?.marginBottom ?? 0;
    if (marginBottom > 0) {
      document.getElementById('footer-container').style.setProperty('padding-bottom', `calc(${marginBottom + (marginBottom / 6)}px)`);
    }

    // Configurações de barra de progresso
    const progressBarContainer = document.getElementById('progress-bar-container');
    if (marginBottom > 0) {
      const progressbarMargin = marginBottom / 5;
      const windowWidth = window.innerWidth;
      const progressBarWidth = windowWidth - (progressbarMargin * 2);
      progressBarContainer.style.setProperty('width', `${progressBarWidth}px`);
      progressBarContainer.style.setProperty('left', `${progressbarMargin}px`);
      progressBarContainer.style.setProperty('right', `${progressbarMargin}px`);
    } else {
      const marginRight = this.format === 'wide' ? 1 : 2;
      const marginLeft = this.format === 'wide' ? 0 : 2;
      progressBarContainer.style.setProperty('left', `${marginLeft}vmax`);
      progressBarContainer.style.setProperty('right', `${marginRight}vmax`);
      progressBarContainer.style.setProperty('width', `calc(100% - ${marginRight + marginLeft}vmax)`);
    }

    if (this.format === 'wide') {
      progressBarContainer.style.setProperty('width', `calc(100% - .5vmax)`);
      progressBarContainer.style.setProperty('margin-left', `-.5vmax`);
    }

    setTimeout(() => {
      document.getElementById('information-bar')?.style.setProperty('align-items', 'center');
      this.adjustContentFontSize();
      this.adjustTitleFontSize();
      this.adjustContentInContainer()
      this.verticalQrCodePosition(marginBottom);
      this.imageDragging();
      this.imageZooming();
      this.resetImageStylesOnDoubleClick();
      this.alignImageWithArrowKeys();
    }, 500);

    setTimeout(() => {
      if (item.bar_visibility === "none") {
        document.querySelector('.content-container').style.setProperty('display', 'none');
        const backgroundMedia = document.getElementById('background_media');
        const mediaElement = backgroundMedia.querySelector('img, video');
        mediaElement.style.setProperty('object-fit', 'fill');
        mediaElement.style.setProperty('width', '100%');
      }
      
    }, 700);

    if (item.qr_code === "news_qrcodes/") {
      document.getElementById('qr-code').style.setProperty('display', 'none');
    }

   

    replaceHtml(object, mapElementId);
    const importants = document.querySelectorAll('.reveal.important');
    importants.forEach(box => {
      box.classList.add('show');
    });
  }

  documentDidMount(ev) {

    if (ev.data.type === 'visibility' && ev?.data?.isVisible) {
      this.readyToStart(ev.data);
    } else if (ev?.data?.type === 'template_data') {
      
      // Ao receber os dados, define o formato da tela de acordo com a proporção
      setTimeout(() => {
        this.screenFormat(ev?.data?.item?.currentPart);
      }, 150);

      // Aguarda 1 segundo para renderizar os dados
      setTimeout(() => {
        this.receivedTemplateData(ev.data);
      }, 600);
    }
  }
}

// Exemplo de uso
window.addEventListener('DOMContentLoaded', () => {
  const renderer = new NewsRenderer();
  // renderer.screenFormat(); // Definir o formato primeiro
  window.addEventListener('message', (ev) => renderer.documentDidMount(ev));
  window.parent.postMessage('load', '*');
});
